<?php
// public_html/px/offersadd/px_client.php

/**
 * PX ping client – shared by:
 *  - test_form.php (manual tests)
 *  - ringba_px_router.php (live Twilio calls)
 */

$config = require __DIR__ . '/config.php';

// Optional helper (ok if it does nothing special)
$helperPath = __DIR__ . '/helper.php';
if (file_exists($helperPath)) {
    require_once $helperPath;
}

/**
 * Core function used by ringba_px_router.php
 *
 * @param string $verticalKey  'roofing' | 'windows' | 'bathroom'
 * @param string $callerId     E.164 caller (From)
 * @param string $zip          5-digit ZIP
 * @param string $subId        GG1 / FB1 / etc.
 * @param string $userAgent    Browser or "Twilio-Voice"
 * @param string $ip           IP address
 *
 * @return array standardized result for router / tester
 *         [
 *            'route_allowed'      => bool,
 *            'reason'             => string,
 *            'destination_number' => string|null,
 *            'transaction_id'     => string|null,
 *            'px_http_code'       => int|null,
 *            'px_raw'             => string,      // raw PX response
 *            'px_curl_error'      => string,
 *            'px_json_sent'       => string       // JSON we sent
 *         ]
 */
function px_ping_call(
    string $verticalKey,
    string $callerId,
    string $zip,
    string $subId,
    string $userAgent,
    string $ip
): array {
    global $config;

    if (empty($config['campaigns'][$verticalKey])) {
        throw new Exception("Unknown vertical: {$verticalKey}");
    }

    $campaign = $config['campaigns'][$verticalKey];

    $apiToken = $campaign['partner_token'];
    $vertical = $campaign['vertical'];
    $offerId  = $campaign['offer_id'];
    $did      = $campaign['did'];

    // -------------------------------
    // Build PX ping payload
    // (matches what you already tested successfully)
    // -------------------------------
    $payload = [
        'ApiToken'    => $apiToken,
        'Vertical'    => $vertical,
        'SubId'       => $subId,
        'Sub2Id'      => '',
        'Sub3Id'      => '',
        'Sub4Id'      => '',
        'Sub5Id'      => '',
        'OfferId'     => (string)$offerId,
        'UserAgent'   => $userAgent,
        'OriginalUrl' => 'https://digimightymedia.com/px-test',

        // you can adjust this if needed: Search / Social / etc.
        'Source'        => 'Search',
        'JornayaLeadId' => '',
        'TrustedForm'   => '',
        'SessionLength' => '30',
        'TcpaText'      => 'I consent to be contacted regarding home improvement services.',

        'ContactData' => [
            'FirstName'      => 'Test',
            'LastName'       => 'Lead',
            'Address'        => '1 Test Street',
            'City'           => 'Test City',
            'State'          => 'NY',
            'ZipCode'        => $zip,
            'EmailAddress'   => 'test@example.com',
            'PhoneNumber'    => $callerId,
            'DayPhoneNumber' => $callerId,
            'IpAddress'      => $ip,
        ],

        'Person' => [
            'FirstName'      => 'Test',
            'LastName'       => 'Lead',
            'BirthDate'      => '1980-01-01',
            'Gender'         => 'Unspecified',
            'BestTimeToCall' => 'Any time',
        ],

        'Custom' => [
            'Field1'  => '',
            'Field2'  => '',
            'Field3'  => '',
            'Field4'  => '',
            'Field5'  => '',
            'Field6'  => '',
            'Field7'  => '',
            'Field8'  => '',
            'Field9'  => '',
            'Field10' => '',
        ],

        // Home payload differs slightly per vertical
        'Home' => build_px_home_block($verticalKey),
    ];

    $json = json_encode($payload);

    // -------------------------------
    // Send to PX ping endpoint
    // -------------------------------
    $ch = curl_init($config['ping_url']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Content-Length: ' . strlen($json),
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json);

    $responseBody = curl_exec($ch);
    $curlErr      = curl_error($ch);
    $httpCode     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // -------------------------------
    // Try to parse PX response (JSON first, then XML fallback)
    // -------------------------------
    $pxData = json_decode($responseBody, true);

    if (!is_array($pxData)) {
        // Fallback: try XML like:
        // <Result><TransactionId>...</TransactionId><Success>true</Success>...</Result>
        $pxData = null;
        if (is_string($responseBody) && trim($responseBody) !== '') {
            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($responseBody);
            if ($xml !== false) {
                $pxData = [
                    'TransactionId' => isset($xml->TransactionId) ? (string)$xml->TransactionId : null,
                    'Success'       => isset($xml->Success)
                        ? filter_var((string)$xml->Success, FILTER_VALIDATE_BOOLEAN)
                        : null,
                    'Payout'        => isset($xml->Payout) ? (float)$xml->Payout : null,
                    'Message'       => isset($xml->Message) ? (string)$xml->Message : null,
                    // keep shape similar to JSON mode
                    'Errors'        => null,
                    'Sold'          => null,
                    'RedirectUrl'   => null,
                    'BuyerGuid'     => isset($xml->BuyerGuid) ? (string)$xml->BuyerGuid : null,
                    'Environment'   => isset($xml->Environment) ? (string)$xml->Environment : null,
                ];
            }
            libxml_clear_errors();
        }
    }

    $success = false;
    $transactionId = null;
    $message = null;

    if (is_array($pxData)) {
        $transactionId = $pxData['TransactionId'] ?? null;

        // Success may be bool or "true"/"false"
        if (array_key_exists('Success', $pxData)) {
            $success = (bool)$pxData['Success'];
        }

        $message = $pxData['Message'] ?? null;
    }

    $routeAllowed = $success && !empty($did);
    $destNumber   = $routeAllowed ? $did : null;
    $reason       = $success ? 'PX Ping accepted' : ($message ?: 'PX Ping rejected');

    return [
        'route_allowed'      => $routeAllowed,
        'reason'             => $reason,
        'destination_number' => $destNumber,
        'transaction_id'     => $transactionId,
        'px_http_code'       => $httpCode,
        'px_raw'             => $responseBody,
        'px_curl_error'      => $curlErr,
        'px_json_sent'       => $json,
    ];
}

/**
 * Build the "Home" block per vertical.
 * Values mirror the ones from your working ping examples.
 */
function build_px_home_block(string $verticalKey): array
{
    switch ($verticalKey) {
        case 'windows':
            return [
                'ProjectType'       => 'New Unit Installed',
                'NumberOfWindows'   => '1 window',
                'WindowStyle'       => 'Bay or Bow',
                'WindowAge'         => 'New (less than 1 year old)',
                'WindowCondition'   => 'Poor',
                'PropertyType'      => 'Residential',
                'Ownership'         => 'Own',
                'AuthorizedToMakeChanges' => 'Yes',
                'PurchaseTimeframe' => 'Immediately',
            ];

        case 'bathroom':
            return [
                'ProjectType'       => 'Repair',
                'Ownership'         => 'Own',
                'AuthorizedToMakeChanges' => 'Yes',
                'PurchaseTimeframe' => 'Timing is flexible',
            ];

        case 'roofing':
        default:
            return [
                'ProjectType'       => 'Repair',
                'Ownership'         => 'Own',
                'AuthorizedToMakeChanges' => 'Yes',
                'PurchaseTimeframe' => 'Timing is flexible',
            ];
    }
}
